'use strict';

/* --------------------------------------------------------------
 orders_modal_layer.js 2016-03-16
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Orders Modal Layer Module
 *
 * This module will open a modal layer for order actions like deleting or changing the oder status.
 *
 * @module Compatibility/orders_modal_layer
 */
gx.compatibility.module('orders_modal_layer', ['xhr', 'fallback'],

/**  @lends module:Compatibility/orders_modal_layer */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES DEFINITION
    // ------------------------------------------------------------------------

    var
    /**
     * Module Selector
     *
     * @var {object}
     */
    $this = $(this),


    /**
     * Modal Selector
     *
     * @type {object}
     */
    $modal = $('#modal_layer_container'),


    /**
     * Checkboxes Selector
     *
     * @type {object}
     */
    $checkboxes = $('.gx-orders-table tr:not(.dataTableHeadingRow) input'),


    /**
     * Default Options
     *
     * @type {object}
     */
    defaults = {
        detail_page: false,
        comment: ''
    },


    /**
     * Final Options
     *
     * @var {object}
     */
    options = $.extend(true, {}, defaults, data),


    /**
     * Module Object
     *
     * @type {object}
     */
    module = {};

    // ------------------------------------------------------------------------
    // PRIVATE FUNCTIONS
    // ------------------------------------------------------------------------

    var _openDeleteDialog = function _openDeleteDialog(event) {

        var $form = $('#delete_confirm_form');
        $form.attr('action', $form.attr('action') + '&oID=' + $this.data('order_id'));

        event.preventDefault();

        var title = jse.core.lang.translate('TEXT_INFO_HEADING_DELETE_ORDER', 'orders').replace('%s', $this.data('order_id'));

        $form.dialog({
            'title': title,
            'modal': true,
            'dialogClass': 'gx-container',
            'buttons': _getModalButtons($form),
            'width': 420
        });
    };

    var _openTrackingCodeDeleteDialog = function _openTrackingCodeDeleteDialog(event) {
        var $form = $('#delete_tracking_code_confirm_form');
        var data_set = jse.libs.fallback._data($(this), 'orders_modal_layer');
        $form.dialog({
            'title': jse.core.lang.translate('TXT_PARCEL_TRACKING_DELETE_BUTTON', 'parcel_services').replace('%s', data_set.tracking_code),
            'modal': true,
            'dialogClass': 'gx-container',
            'buttons': [{
                'text': jse.core.lang.translate('close', 'buttons'),
                'class': 'btn',
                'click': function click() {
                    $(this).dialog('close');
                }
            }, {
                'text': jse.core.lang.translate('delete', 'buttons'),
                'class': 'btn btn-primary',
                'click': function click() {
                    $(this).dialog('close');

                    var url = 'request_port.php?module=ParcelServices&action=delete_tracking_code';

                    jse.libs.xhr.post({
                        'url': url,
                        'data': {
                            'tracking_code_id': data_set.tracking_code_id,
                            'order_id': data_set.order_id,
                            'page_token': data_set.page_token
                        }
                    }).done(function (response) {
                        $('#tracking_code_wrapper > .frame-content > table').html(response.html);
                    });
                }
            }],
            'width': 420
        });
    };

    var _openMultiDeleteDialog = function _openMultiDeleteDialog(event) {

        var $form = $('#multi_delete_confirm_form'),
            orderId = 0;

        event.preventDefault();

        if ($checkboxes.filter(':checked').length === 0) {
            return false;
        }

        _readSelectedOrders($form);

        $form.attr('action', $form.attr('action') + '&oID=' + $this.data('order_id'));

        $form.dialog({
            'title': jse.core.lang.translate('TEXT_INFO_HEADING_MULTI_DELETE_ORDER', 'orders').replace('%s', $this.data('order_id')),
            'modal': true,
            'dialogClass': 'gx-container',
            'buttons': _getModalButtons($form),
            'width': 420
        });
    };

    var _openMultiCancelDialog = function _openMultiCancelDialog(event) {
        var $form = $('#multi_cancel_confirm_form');
        event.preventDefault();

        if (options.detail_page) {
            // Orders detail page
            $form.append('<input type="hidden" name="gm_multi_status[]" value="' + options.order_id + '" />');
            $form.find('.selected_orders').text(options.order_id);
            $form.find('textarea[name="gm_comments"]').html(options.comment);
        } else {
            // Orders page
            if ($checkboxes.filter(':checked').length === 0) {
                return false;
            }
            _readSelectedOrders($form);
        }

        $form.attr('action', $form.attr('action') + '?oID=' + $this.data('order_id') + '&origin=old_orders_overview');

        $form.dialog({
            'title': jse.core.lang.translate('TEXT_INFO_HEADING_MULTI_CANCEL_ORDER', 'orders').replace('%s', $this.data('order_id')),
            'modal': true,
            'dialogClass': 'gx-container',
            'buttons': _getModalButtons($form),
            'width': 420
        });
    };

    var _openUpdateOrdersStatusDialog = function _openUpdateOrdersStatusDialog(event) {
        var $form = $('#update_orders_status_form');

        event.preventDefault();

        if (options.detail_page) {
            // Orders detail page
            $form.append('<input type="hidden" name="gm_multi_status[]" value="' + options.order_id + '" />');
            $form.find('.selected_orders').text(options.order_id);
            $form.find('textarea[name="gm_comments"]').html(options.comment);
        } else {
            // Orders page
            if ($checkboxes.filter(':checked').length === 0) {
                return false;
            }
            _readSelectedOrders($form);
        }

        $form.dialog({
            'title': jse.core.lang.translate('HEADING_GM_STATUS', 'orders'),
            'modal': true,
            'dialogClass': 'gx-container',
            'buttons': _getModalButtons($form),
            'width': 580
        });
    };

    var _openTrackingCodeDialog = function _openTrackingCodeDialog(event) {

        var $form = $('#add_tracking_code_form');

        event.preventDefault();
        $form.dialog({
            'title': jse.core.lang.translate('TXT_PARCEL_TRACKING_HEADING', 'parcel_services').replace('%s', $this.data('order_id')),
            'modal': true,
            'dialogClass': 'gx-container',
            'buttons': _getModalButtons($form),
            'width': 420
        });
    };

    var _openEmailInvoiceDialog = function _openEmailInvoiceDialog(event) {

        var $modal = $('.email-invoice-modal-body');
        var url = jse.core.config.get('appUrl') + '/admin/admin.php';
        var data = {
            id: options.order_id,
            do: 'OrdersModalsAjax/GetEmailInvoiceSubject',
            pageToken: jse.core.config.get('pageToken')
        };
        var invoiceNumbersHtml = '';

        $('.email-invoice-form').show();
        $('.email-invoice-success').hide();

        if ($modal.find('.message_stack_container').length < 1) {
            $modal.find('.alert').wrap('<div class="message_stack_container"></div>');
        }

        $modal.find('.customer-info').text('"' + options.name + '"');
        $modal.find('.email-address').val(options.email);

        $modal.data('orderId', options.order_id);

        $.ajax({ url: url, data: data, dataType: 'json' }).done(function (response) {
            $modal.attr('data-gx-widget', 'single_checkbox');

            $modal.find('.subject').val(response.subject);
            if (response.invoiceIdExists) {
                $modal.find('.invoice-num-info').addClass('hidden');
                $modal.find('.no-invoice').removeClass('hidden');
                $modal.find('.email-invoice-form .message_stack_container').removeClass('hidden');
            } else {
                $modal.find('.invoice-num-info').removeClass('hidden');
                $modal.find('.no-invoice').addClass('hidden');
                $modal.find('.email-invoice-form .message_stack_container').addClass('hidden');
            }

            if (Object.keys(response.invoiceNumbers).length <= 1) {
                $modal.find('.invoice-numbers').addClass('hidden');
            } else {
                $modal.find('.invoice-numbers').removeClass('hidden');
            }

            for (var invoiceId in response.invoiceNumbers) {
                invoiceNumbersHtml += '<p><input type="checkbox" name="invoice_ids[]" value="' + invoiceId + '" checked="checked" class="invoice-numbers-checkbox" /> ' + response.invoiceNumbers[invoiceId] + '</p>';
            }

            $modal.find('.invoice-numbers-checkboxes').html(invoiceNumbersHtml);

            gx.widgets.init($modal);

            $modal.dialog({
                'title': jse.core.lang.translate('TITLE_INVOICE', 'gm_order_menu'),
                'modal': true,
                'dialogClass': 'gx-container',
                'buttons': _getModalButtons($modal),
                'width': 600
            });

            $modal.find('.invoice-numbers-checkbox').on('change', function () {
                _onChangeEmailInvoiceCheckbox($modal);
            });
        });

        event.preventDefault();
    };

    /**
     * On Email Invoice Checkbox Change
     *
     * Disable send button if all invoice number checkboxes are unchecked. Otherwise enable the send button again.
     */
    var _onChangeEmailInvoiceCheckbox = function _onChangeEmailInvoiceCheckbox($modal) {
        if ($modal.find('.invoice-numbers-checkbox').length > 0) {
            if ($modal.find('.invoice-numbers-checkbox:checked').length > 0) {
                $('.btn-send-invoice-mail').prop('disabled', false);
            } else {
                $('.btn-send-invoice-mail').prop('disabled', true);
            }
        } else {
            $('.btn-send-invoice-mail').prop('disabled', false);
        }
    };

    var _getModalButtons = function _getModalButtons($form) {
        var buttons = [{
            'text': jse.core.lang.translate('close', 'buttons'),
            'class': 'btn',
            'click': function click() {
                $(this).dialog('close');
            }
        }];
        switch (options.action) {
            case 'delete':
            case 'multi_delete':
                buttons.push({
                    'text': jse.core.lang.translate('delete', 'buttons'),
                    'class': 'btn btn-primary',
                    'click': function click() {
                        $form.submit();
                    }
                });
                break;
            case 'add_tracking_code':
                buttons.push({
                    'text': jse.core.lang.translate('add', 'buttons'),
                    'class': 'btn btn-primary',
                    'click': function click(event) {
                        _addTrackingCodeFromOverview(event);
                    }
                });
                break;
            case 'update_orders_status':
                buttons.push({
                    'text': jse.core.lang.translate('execute', 'buttons'),
                    'class': 'btn btn-primary',
                    'click': function click(event) {
                        $form.submit();
                    }
                });
                break;
            case 'multi_cancel':
                buttons.push({
                    'text': jse.core.lang.translate('send', 'buttons'),
                    'class': 'btn btn-primary',
                    'click': function click(event) {
                        //event.preventDefault();
                        //gm_cancel('gm_send_order.php', '&type=cancel', 'CANCEL');
                        $form.submit();
                    }
                });
                break;
            case 'email_invoice':
                buttons.push({
                    'text': jse.core.lang.translate('send', 'buttons'),
                    'class': 'btn btn-primary btn-send-invoice-mail',
                    'click': function click(event) {
                        event.preventDefault();

                        var url = 'gm_pdf_order.php?oID=' + $form.data('orderId') + '&type=invoice&mail=1&gm_quick_mail=1';
                        var data = $form.find('form').serialize();

                        $.ajax({ url: url, data: data, type: 'POST', dataType: 'html' }).done(function (response) {
                            $('.email-invoice-form').hide();
                            $('.email-invoice-success').show();

                            $('.btn-send-invoice-mail').hide();
                        });
                    }
                });
        }

        return buttons;
    };

    var _addTrackingCodeFromOverview = function _addTrackingCodeFromOverview(event) {
        event.stopPropagation();

        var tracking_code = $('#parcel_service_tracking_code').val();
        if (tracking_code === '') {
            return false;
        }

        $.ajax({
            'type': 'POST',
            'url': 'request_port.php?module=ParcelServices&action=add_tracking_code',
            'timeout': 30000,
            'dataType': 'json',
            'context': this,
            'data': {

                'tracking_code': tracking_code,
                'service_id': $('#parcel_services_dropdown option:selected').val(),
                'order_id': $this.data('order_id'),
                'page_token': $('.page_token').val()
            },
            done: function done() {
                document.location.reload();
            }
        });

        return false;
    };

    var _readSelectedOrders = function _readSelectedOrders($form) {
        var orderIds = [];

        $checkboxes.filter(':checked').each(function () {
            $form.append('<input type="hidden" name="gm_multi_status[]" value="' + $(this).val() + '" />');

            orderIds.push($(this).val());
        });

        $form.find('.selected_orders').text(orderIds.join(', '));
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        switch (options.action) {
            case 'delete':
                $this.on('click', _openDeleteDialog);
                break;
            case 'multi_delete':
                $this.on('click', _openMultiDeleteDialog);
                break;
            case 'add_tracking_code':
                $this.on('click', _openTrackingCodeDialog);
                break;
            case 'update_orders_status':
                $this.on('click', _openUpdateOrdersStatusDialog);
                break;
            case 'multi_cancel':
                $this.on('click', _openMultiCancelDialog);
                break;
            case 'email_invoice':
                $this.on('click', _openEmailInvoiceDialog);
                break;
        }

        if (options.container === 'tracking_code_wrapper') {
            $this.on('click', '.btn-delete', _openTrackingCodeDeleteDialog);
        }

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIm9yZGVycy9vcmRlcnNfbW9kYWxfbGF5ZXIuanMiXSwibmFtZXMiOlsiZ3giLCJjb21wYXRpYmlsaXR5IiwibW9kdWxlIiwiZGF0YSIsIiR0aGlzIiwiJCIsIiRtb2RhbCIsIiRjaGVja2JveGVzIiwiZGVmYXVsdHMiLCJkZXRhaWxfcGFnZSIsImNvbW1lbnQiLCJvcHRpb25zIiwiZXh0ZW5kIiwiX29wZW5EZWxldGVEaWFsb2ciLCJldmVudCIsIiRmb3JtIiwiYXR0ciIsInByZXZlbnREZWZhdWx0IiwidGl0bGUiLCJqc2UiLCJjb3JlIiwibGFuZyIsInRyYW5zbGF0ZSIsInJlcGxhY2UiLCJkaWFsb2ciLCJfZ2V0TW9kYWxCdXR0b25zIiwiX29wZW5UcmFja2luZ0NvZGVEZWxldGVEaWFsb2ciLCJkYXRhX3NldCIsImxpYnMiLCJmYWxsYmFjayIsIl9kYXRhIiwidHJhY2tpbmdfY29kZSIsInVybCIsInhociIsInBvc3QiLCJ0cmFja2luZ19jb2RlX2lkIiwib3JkZXJfaWQiLCJwYWdlX3Rva2VuIiwiZG9uZSIsInJlc3BvbnNlIiwiaHRtbCIsIl9vcGVuTXVsdGlEZWxldGVEaWFsb2ciLCJvcmRlcklkIiwiZmlsdGVyIiwibGVuZ3RoIiwiX3JlYWRTZWxlY3RlZE9yZGVycyIsIl9vcGVuTXVsdGlDYW5jZWxEaWFsb2ciLCJhcHBlbmQiLCJmaW5kIiwidGV4dCIsIl9vcGVuVXBkYXRlT3JkZXJzU3RhdHVzRGlhbG9nIiwiX29wZW5UcmFja2luZ0NvZGVEaWFsb2ciLCJfb3BlbkVtYWlsSW52b2ljZURpYWxvZyIsImNvbmZpZyIsImdldCIsImlkIiwiZG8iLCJwYWdlVG9rZW4iLCJpbnZvaWNlTnVtYmVyc0h0bWwiLCJzaG93IiwiaGlkZSIsIndyYXAiLCJuYW1lIiwidmFsIiwiZW1haWwiLCJhamF4IiwiZGF0YVR5cGUiLCJzdWJqZWN0IiwiaW52b2ljZUlkRXhpc3RzIiwiYWRkQ2xhc3MiLCJyZW1vdmVDbGFzcyIsIk9iamVjdCIsImtleXMiLCJpbnZvaWNlTnVtYmVycyIsImludm9pY2VJZCIsIndpZGdldHMiLCJpbml0Iiwib24iLCJfb25DaGFuZ2VFbWFpbEludm9pY2VDaGVja2JveCIsInByb3AiLCJidXR0b25zIiwiYWN0aW9uIiwicHVzaCIsInN1Ym1pdCIsIl9hZGRUcmFja2luZ0NvZGVGcm9tT3ZlcnZpZXciLCJzZXJpYWxpemUiLCJ0eXBlIiwic3RvcFByb3BhZ2F0aW9uIiwiZG9jdW1lbnQiLCJsb2NhdGlvbiIsInJlbG9hZCIsIm9yZGVySWRzIiwiZWFjaCIsImpvaW4iLCJjb250YWluZXIiXSwibWFwcGluZ3MiOiI7O0FBQUE7Ozs7Ozs7Ozs7QUFVQTs7Ozs7OztBQU9BQSxHQUFHQyxhQUFILENBQWlCQyxNQUFqQixDQUNJLG9CQURKLEVBR0ksQ0FBQyxLQUFELEVBQVEsVUFBUixDQUhKOztBQUtJOztBQUVBLFVBQVVDLElBQVYsRUFBZ0I7O0FBRVo7O0FBRUE7QUFDQTtBQUNBOztBQUVBO0FBQ0k7Ozs7O0FBS0FDLFlBQVFDLEVBQUUsSUFBRixDQU5aOzs7QUFRSTs7Ozs7QUFLQUMsYUFBU0QsRUFBRSx3QkFBRixDQWJiOzs7QUFlSTs7Ozs7QUFLQUUsa0JBQWNGLEVBQUUscURBQUYsQ0FwQmxCOzs7QUFzQkk7Ozs7O0FBS0FHLGVBQVc7QUFDUEMscUJBQWEsS0FETjtBQUVQQyxpQkFBUztBQUZGLEtBM0JmOzs7QUFnQ0k7Ozs7O0FBS0FDLGNBQVVOLEVBQUVPLE1BQUYsQ0FBUyxJQUFULEVBQWUsRUFBZixFQUFtQkosUUFBbkIsRUFBNkJMLElBQTdCLENBckNkOzs7QUF1Q0k7Ozs7O0FBS0FELGFBQVMsRUE1Q2I7O0FBOENBO0FBQ0E7QUFDQTs7QUFFQSxRQUFJVyxvQkFBb0IsU0FBcEJBLGlCQUFvQixDQUFVQyxLQUFWLEVBQWlCOztBQUVyQyxZQUFJQyxRQUFRVixFQUFFLHNCQUFGLENBQVo7QUFDQVUsY0FBTUMsSUFBTixDQUFXLFFBQVgsRUFBcUJELE1BQU1DLElBQU4sQ0FBVyxRQUFYLElBQXVCLE9BQXZCLEdBQWlDWixNQUFNRCxJQUFOLENBQVcsVUFBWCxDQUF0RDs7QUFFQVcsY0FBTUcsY0FBTjs7QUFFQSxZQUFJQyxRQUFRQyxJQUFJQyxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixnQ0FBeEIsRUFBMEQsUUFBMUQsRUFDUEMsT0FETyxDQUNDLElBREQsRUFDT25CLE1BQU1ELElBQU4sQ0FBVyxVQUFYLENBRFAsQ0FBWjs7QUFHQVksY0FBTVMsTUFBTixDQUFhO0FBQ1QscUJBQVNOLEtBREE7QUFFVCxxQkFBUyxJQUZBO0FBR1QsMkJBQWUsY0FITjtBQUlULHVCQUFXTyxpQkFBaUJWLEtBQWpCLENBSkY7QUFLVCxxQkFBUztBQUxBLFNBQWI7QUFRSCxLQWxCRDs7QUFvQkEsUUFBSVcsZ0NBQWdDLFNBQWhDQSw2QkFBZ0MsQ0FBVVosS0FBVixFQUFpQjtBQUNqRCxZQUFJQyxRQUFRVixFQUFFLG9DQUFGLENBQVo7QUFDQSxZQUFJc0IsV0FBV1IsSUFBSVMsSUFBSixDQUFTQyxRQUFULENBQWtCQyxLQUFsQixDQUF3QnpCLEVBQUUsSUFBRixDQUF4QixFQUFpQyxvQkFBakMsQ0FBZjtBQUNBVSxjQUFNUyxNQUFOLENBQWE7QUFDVCxxQkFBU0wsSUFBSUMsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsbUNBQXhCLEVBQTZELGlCQUE3RCxFQUNKQyxPQURJLENBRUQsSUFGQyxFQUVLSSxTQUFTSSxhQUZkLENBREE7QUFJVCxxQkFBUyxJQUpBO0FBS1QsMkJBQWUsY0FMTjtBQU1ULHVCQUFXLENBQ1A7QUFDSSx3QkFBUVosSUFBSUMsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsT0FBeEIsRUFBaUMsU0FBakMsQ0FEWjtBQUVJLHlCQUFTLEtBRmI7QUFHSSx5QkFBUyxpQkFBWTtBQUNqQmpCLHNCQUFFLElBQUYsRUFBUW1CLE1BQVIsQ0FBZSxPQUFmO0FBQ0g7QUFMTCxhQURPLEVBUVA7QUFDSSx3QkFBUUwsSUFBSUMsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsUUFBeEIsRUFBa0MsU0FBbEMsQ0FEWjtBQUVJLHlCQUFTLGlCQUZiO0FBR0kseUJBQVMsaUJBQVk7QUFDakJqQixzQkFBRSxJQUFGLEVBQVFtQixNQUFSLENBQWUsT0FBZjs7QUFFQSx3QkFBSVEsTUFBTSxvRUFBVjs7QUFFQWIsd0JBQUlTLElBQUosQ0FBU0ssR0FBVCxDQUFhQyxJQUFiLENBQWtCO0FBQ2QsK0JBQU9GLEdBRE87QUFFZCxnQ0FBUTtBQUNKLGdEQUFvQkwsU0FBU1EsZ0JBRHpCO0FBRUosd0NBQVlSLFNBQVNTLFFBRmpCO0FBR0osMENBQWNULFNBQVNVO0FBSG5CO0FBRk0scUJBQWxCLEVBT0dDLElBUEgsQ0FPUSxVQUFVQyxRQUFWLEVBQW9CO0FBQ3hCbEMsMEJBQUUsaURBQUYsRUFBcURtQyxJQUFyRCxDQUEwREQsU0FBU0MsSUFBbkU7QUFDSCxxQkFURDtBQVVIO0FBbEJMLGFBUk8sQ0FORjtBQW1DVCxxQkFBUztBQW5DQSxTQUFiO0FBc0NILEtBekNEOztBQTJDQSxRQUFJQyx5QkFBeUIsU0FBekJBLHNCQUF5QixDQUFVM0IsS0FBVixFQUFpQjs7QUFFMUMsWUFBSUMsUUFBUVYsRUFBRSw0QkFBRixDQUFaO0FBQUEsWUFDSXFDLFVBQVUsQ0FEZDs7QUFHQTVCLGNBQU1HLGNBQU47O0FBRUEsWUFBSVYsWUFBWW9DLE1BQVosQ0FBbUIsVUFBbkIsRUFBK0JDLE1BQS9CLEtBQTBDLENBQTlDLEVBQWlEO0FBQzdDLG1CQUFPLEtBQVA7QUFDSDs7QUFFREMsNEJBQW9COUIsS0FBcEI7O0FBRUFBLGNBQU1DLElBQU4sQ0FBVyxRQUFYLEVBQXFCRCxNQUFNQyxJQUFOLENBQVcsUUFBWCxJQUF1QixPQUF2QixHQUFpQ1osTUFBTUQsSUFBTixDQUFXLFVBQVgsQ0FBdEQ7O0FBRUFZLGNBQU1TLE1BQU4sQ0FBYTtBQUNULHFCQUFTTCxJQUFJQyxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixzQ0FBeEIsRUFBZ0UsUUFBaEUsRUFDSkMsT0FESSxDQUNJLElBREosRUFFRG5CLE1BQU1ELElBQU4sQ0FBVyxVQUFYLENBRkMsQ0FEQTtBQUlULHFCQUFTLElBSkE7QUFLVCwyQkFBZSxjQUxOO0FBTVQsdUJBQVdzQixpQkFBaUJWLEtBQWpCLENBTkY7QUFPVCxxQkFBUztBQVBBLFNBQWI7QUFTSCxLQXhCRDs7QUEwQkEsUUFBSStCLHlCQUF5QixTQUF6QkEsc0JBQXlCLENBQVVoQyxLQUFWLEVBQWlCO0FBQzFDLFlBQUlDLFFBQVFWLEVBQUUsNEJBQUYsQ0FBWjtBQUNBUyxjQUFNRyxjQUFOOztBQUVBLFlBQUlOLFFBQVFGLFdBQVosRUFBeUI7QUFDckI7QUFDQU0sa0JBQU1nQyxNQUFOLENBQWEsMERBQTBEcEMsUUFBUXlCLFFBQWxFLEdBQ1QsTUFESjtBQUVBckIsa0JBQU1pQyxJQUFOLENBQVcsa0JBQVgsRUFBK0JDLElBQS9CLENBQW9DdEMsUUFBUXlCLFFBQTVDO0FBQ0FyQixrQkFBTWlDLElBQU4sQ0FBVyw4QkFBWCxFQUEyQ1IsSUFBM0MsQ0FBZ0Q3QixRQUFRRCxPQUF4RDtBQUNILFNBTkQsTUFNTztBQUNIO0FBQ0EsZ0JBQUlILFlBQVlvQyxNQUFaLENBQW1CLFVBQW5CLEVBQStCQyxNQUEvQixLQUEwQyxDQUE5QyxFQUFpRDtBQUM3Qyx1QkFBTyxLQUFQO0FBQ0g7QUFDREMsZ0NBQW9COUIsS0FBcEI7QUFDSDs7QUFFREEsY0FBTUMsSUFBTixDQUFXLFFBQVgsRUFBcUJELE1BQU1DLElBQU4sQ0FBVyxRQUFYLElBQXVCLE9BQXZCLEdBQWlDWixNQUFNRCxJQUFOLENBQVcsVUFBWCxDQUFqQyxHQUEwRCw2QkFBL0U7O0FBRUFZLGNBQU1TLE1BQU4sQ0FBYTtBQUNULHFCQUFTTCxJQUFJQyxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixzQ0FBeEIsRUFBZ0UsUUFBaEUsRUFDSkMsT0FESSxDQUNJLElBREosRUFFRG5CLE1BQU1ELElBQU4sQ0FBVyxVQUFYLENBRkMsQ0FEQTtBQUlULHFCQUFTLElBSkE7QUFLVCwyQkFBZSxjQUxOO0FBTVQsdUJBQVdzQixpQkFBaUJWLEtBQWpCLENBTkY7QUFPVCxxQkFBUztBQVBBLFNBQWI7QUFTSCxLQTdCRDs7QUErQkEsUUFBSW1DLGdDQUFnQyxTQUFoQ0EsNkJBQWdDLENBQVVwQyxLQUFWLEVBQWlCO0FBQ2pELFlBQUlDLFFBQVFWLEVBQUUsNEJBQUYsQ0FBWjs7QUFFQVMsY0FBTUcsY0FBTjs7QUFFQSxZQUFJTixRQUFRRixXQUFaLEVBQXlCO0FBQ3JCO0FBQ0FNLGtCQUFNZ0MsTUFBTixDQUFhLDBEQUEwRHBDLFFBQVF5QixRQUFsRSxHQUNULE1BREo7QUFFQXJCLGtCQUFNaUMsSUFBTixDQUFXLGtCQUFYLEVBQStCQyxJQUEvQixDQUFvQ3RDLFFBQVF5QixRQUE1QztBQUNBckIsa0JBQU1pQyxJQUFOLENBQVcsOEJBQVgsRUFBMkNSLElBQTNDLENBQWdEN0IsUUFBUUQsT0FBeEQ7QUFDSCxTQU5ELE1BTU87QUFDSDtBQUNBLGdCQUFJSCxZQUFZb0MsTUFBWixDQUFtQixVQUFuQixFQUErQkMsTUFBL0IsS0FBMEMsQ0FBOUMsRUFBaUQ7QUFDN0MsdUJBQU8sS0FBUDtBQUNIO0FBQ0RDLGdDQUFvQjlCLEtBQXBCO0FBQ0g7O0FBRURBLGNBQU1TLE1BQU4sQ0FBYTtBQUNULHFCQUFTTCxJQUFJQyxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixtQkFBeEIsRUFBNkMsUUFBN0MsQ0FEQTtBQUVULHFCQUFTLElBRkE7QUFHVCwyQkFBZSxjQUhOO0FBSVQsdUJBQVdHLGlCQUFpQlYsS0FBakIsQ0FKRjtBQUtULHFCQUFTO0FBTEEsU0FBYjtBQU9ILEtBMUJEOztBQTRCQSxRQUFJb0MsMEJBQTBCLFNBQTFCQSx1QkFBMEIsQ0FBVXJDLEtBQVYsRUFBaUI7O0FBRTNDLFlBQUlDLFFBQVFWLEVBQUUseUJBQUYsQ0FBWjs7QUFFQVMsY0FBTUcsY0FBTjtBQUNBRixjQUFNUyxNQUFOLENBQWE7QUFDVCxxQkFBU0wsSUFBSUMsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsNkJBQXhCLEVBQXVELGlCQUF2RCxFQUNKQyxPQURJLENBQ0ksSUFESixFQUNVbkIsTUFBTUQsSUFBTixDQUFXLFVBQVgsQ0FEVixDQURBO0FBR1QscUJBQVMsSUFIQTtBQUlULDJCQUFlLGNBSk47QUFLVCx1QkFBV3NCLGlCQUFpQlYsS0FBakIsQ0FMRjtBQU1ULHFCQUFTO0FBTkEsU0FBYjtBQVNILEtBZEQ7O0FBZ0JBLFFBQUlxQywwQkFBMEIsU0FBMUJBLHVCQUEwQixDQUFVdEMsS0FBVixFQUFpQjs7QUFFM0MsWUFBTVIsU0FBU0QsRUFBRSwyQkFBRixDQUFmO0FBQ0EsWUFBTTJCLE1BQU1iLElBQUlDLElBQUosQ0FBU2lDLE1BQVQsQ0FBZ0JDLEdBQWhCLENBQW9CLFFBQXBCLElBQWdDLGtCQUE1QztBQUNBLFlBQU1uRCxPQUFPO0FBQ1RvRCxnQkFBSTVDLFFBQVF5QixRQURIO0FBRVRvQixnQkFBSSx5Q0FGSztBQUdUQyx1QkFBV3RDLElBQUlDLElBQUosQ0FBU2lDLE1BQVQsQ0FBZ0JDLEdBQWhCLENBQW9CLFdBQXBCO0FBSEYsU0FBYjtBQUtBLFlBQUlJLHFCQUFxQixFQUF6Qjs7QUFFQXJELFVBQUUscUJBQUYsRUFBeUJzRCxJQUF6QjtBQUNBdEQsVUFBRSx3QkFBRixFQUE0QnVELElBQTVCOztBQUVBLFlBQUl0RCxPQUFPMEMsSUFBUCxDQUFZLDBCQUFaLEVBQXdDSixNQUF4QyxHQUFpRCxDQUFyRCxFQUF3RDtBQUNwRHRDLG1CQUFPMEMsSUFBUCxDQUFZLFFBQVosRUFBc0JhLElBQXRCLENBQTJCLDZDQUEzQjtBQUNIOztBQUVEdkQsZUFBTzBDLElBQVAsQ0FBWSxnQkFBWixFQUE4QkMsSUFBOUIsT0FBdUN0QyxRQUFRbUQsSUFBL0M7QUFDQXhELGVBQU8wQyxJQUFQLENBQVksZ0JBQVosRUFBOEJlLEdBQTlCLENBQWtDcEQsUUFBUXFELEtBQTFDOztBQUVBMUQsZUFDS0gsSUFETCxDQUNVLFNBRFYsRUFDcUJRLFFBQVF5QixRQUQ3Qjs7QUFHQS9CLFVBQUU0RCxJQUFGLENBQU8sRUFBQ2pDLFFBQUQsRUFBTTdCLFVBQU4sRUFBWStELFVBQVUsTUFBdEIsRUFBUCxFQUFzQzVCLElBQXRDLENBQTJDLFVBQUNDLFFBQUQsRUFBYztBQUNyRGpDLG1CQUFPVSxJQUFQLENBQVksZ0JBQVosRUFBOEIsaUJBQTlCOztBQUVBVixtQkFBTzBDLElBQVAsQ0FBWSxVQUFaLEVBQXdCZSxHQUF4QixDQUE0QnhCLFNBQVM0QixPQUFyQztBQUNBLGdCQUFJNUIsU0FBUzZCLGVBQWIsRUFBOEI7QUFDMUI5RCx1QkFBTzBDLElBQVAsQ0FBWSxtQkFBWixFQUFpQ3FCLFFBQWpDLENBQTBDLFFBQTFDO0FBQ0EvRCx1QkFBTzBDLElBQVAsQ0FBWSxhQUFaLEVBQTJCc0IsV0FBM0IsQ0FBdUMsUUFBdkM7QUFDQWhFLHVCQUFPMEMsSUFBUCxDQUFZLDhDQUFaLEVBQTREc0IsV0FBNUQsQ0FBd0UsUUFBeEU7QUFDSCxhQUpELE1BSU87QUFDSGhFLHVCQUFPMEMsSUFBUCxDQUFZLG1CQUFaLEVBQWlDc0IsV0FBakMsQ0FBNkMsUUFBN0M7QUFDQWhFLHVCQUFPMEMsSUFBUCxDQUFZLGFBQVosRUFBMkJxQixRQUEzQixDQUFvQyxRQUFwQztBQUNBL0QsdUJBQU8wQyxJQUFQLENBQVksOENBQVosRUFBNERxQixRQUE1RCxDQUFxRSxRQUFyRTtBQUNIOztBQUVELGdCQUFJRSxPQUFPQyxJQUFQLENBQVlqQyxTQUFTa0MsY0FBckIsRUFBcUM3QixNQUFyQyxJQUErQyxDQUFuRCxFQUFzRDtBQUNsRHRDLHVCQUFPMEMsSUFBUCxDQUFZLGtCQUFaLEVBQWdDcUIsUUFBaEMsQ0FBeUMsUUFBekM7QUFDSCxhQUZELE1BRU87QUFDSC9ELHVCQUFPMEMsSUFBUCxDQUFZLGtCQUFaLEVBQWdDc0IsV0FBaEMsQ0FBNEMsUUFBNUM7QUFDSDs7QUFFRCxpQkFBSyxJQUFJSSxTQUFULElBQXNCbkMsU0FBU2tDLGNBQS9CLEVBQStDO0FBQzNDZixzQ0FDSSwyREFBMkRnQixTQUEzRCxHQUNFLDBEQURGLEdBRUVuQyxTQUFTa0MsY0FBVCxDQUF3QkMsU0FBeEIsQ0FGRixHQUV1QyxNQUgzQztBQUlIOztBQUVEcEUsbUJBQU8wQyxJQUFQLENBQVksNkJBQVosRUFBMkNSLElBQTNDLENBQWdEa0Isa0JBQWhEOztBQUVBMUQsZUFBRzJFLE9BQUgsQ0FBV0MsSUFBWCxDQUFnQnRFLE1BQWhCOztBQUVBQSxtQkFBT2tCLE1BQVAsQ0FBYztBQUNWLHlCQUFTTCxJQUFJQyxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixlQUF4QixFQUF5QyxlQUF6QyxDQURDO0FBRVYseUJBQVMsSUFGQztBQUdWLCtCQUFlLGNBSEw7QUFJViwyQkFBV0csaUJBQWlCbkIsTUFBakIsQ0FKRDtBQUtWLHlCQUFTO0FBTEMsYUFBZDs7QUFRQUEsbUJBQU8wQyxJQUFQLENBQVksMkJBQVosRUFBeUM2QixFQUF6QyxDQUE0QyxRQUE1QyxFQUFzRCxZQUFZO0FBQzlEQyw4Q0FBOEJ4RSxNQUE5QjtBQUNILGFBRkQ7QUFHSCxTQTFDRDs7QUE0Q0FRLGNBQU1HLGNBQU47QUFDSCxLQXJFRDs7QUF1RUE7Ozs7O0FBS0EsUUFBSTZELGdDQUFnQyxTQUFoQ0EsNkJBQWdDLENBQVV4RSxNQUFWLEVBQWtCO0FBQ2xELFlBQUlBLE9BQU8wQyxJQUFQLENBQVksMkJBQVosRUFBeUNKLE1BQXpDLEdBQWtELENBQXRELEVBQXlEO0FBQ3JELGdCQUFJdEMsT0FBTzBDLElBQVAsQ0FBWSxtQ0FBWixFQUFpREosTUFBakQsR0FBMEQsQ0FBOUQsRUFBaUU7QUFDN0R2QyxrQkFBRSx3QkFBRixFQUE0QjBFLElBQTVCLENBQWlDLFVBQWpDLEVBQTZDLEtBQTdDO0FBQ0gsYUFGRCxNQUVPO0FBQ0gxRSxrQkFBRSx3QkFBRixFQUE0QjBFLElBQTVCLENBQWlDLFVBQWpDLEVBQTZDLElBQTdDO0FBQ0g7QUFDSixTQU5ELE1BTU87QUFDSDFFLGNBQUUsd0JBQUYsRUFBNEIwRSxJQUE1QixDQUFpQyxVQUFqQyxFQUE2QyxLQUE3QztBQUNIO0FBQ0osS0FWRDs7QUFZQSxRQUFJdEQsbUJBQW1CLFNBQW5CQSxnQkFBbUIsQ0FBVVYsS0FBVixFQUFpQjtBQUNwQyxZQUFJaUUsVUFBVSxDQUNWO0FBQ0ksb0JBQVE3RCxJQUFJQyxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixPQUF4QixFQUFpQyxTQUFqQyxDQURaO0FBRUkscUJBQVMsS0FGYjtBQUdJLHFCQUFTLGlCQUFZO0FBQ2pCakIsa0JBQUUsSUFBRixFQUFRbUIsTUFBUixDQUFlLE9BQWY7QUFDSDtBQUxMLFNBRFUsQ0FBZDtBQVNBLGdCQUFRYixRQUFRc0UsTUFBaEI7QUFDSSxpQkFBSyxRQUFMO0FBQ0EsaUJBQUssY0FBTDtBQUNJRCx3QkFBUUUsSUFBUixDQUFhO0FBQ1QsNEJBQVEvRCxJQUFJQyxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixRQUF4QixFQUFrQyxTQUFsQyxDQURDO0FBRVQsNkJBQVMsaUJBRkE7QUFHVCw2QkFBUyxpQkFBWTtBQUNqQlAsOEJBQU1vRSxNQUFOO0FBQ0g7QUFMUSxpQkFBYjtBQU9BO0FBQ0osaUJBQUssbUJBQUw7QUFDSUgsd0JBQVFFLElBQVIsQ0FBYTtBQUNULDRCQUFRL0QsSUFBSUMsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsS0FBeEIsRUFBK0IsU0FBL0IsQ0FEQztBQUVULDZCQUFTLGlCQUZBO0FBR1QsNkJBQVMsZUFBVVIsS0FBVixFQUFpQjtBQUN0QnNFLHFEQUE2QnRFLEtBQTdCO0FBQ0g7QUFMUSxpQkFBYjtBQU9BO0FBQ0osaUJBQUssc0JBQUw7QUFDSWtFLHdCQUFRRSxJQUFSLENBQWE7QUFDVCw0QkFBUS9ELElBQUlDLElBQUosQ0FBU0MsSUFBVCxDQUFjQyxTQUFkLENBQXdCLFNBQXhCLEVBQW1DLFNBQW5DLENBREM7QUFFVCw2QkFBUyxpQkFGQTtBQUdULDZCQUFTLGVBQVVSLEtBQVYsRUFBaUI7QUFDdEJDLDhCQUFNb0UsTUFBTjtBQUNIO0FBTFEsaUJBQWI7QUFPQTtBQUNKLGlCQUFLLGNBQUw7QUFDSUgsd0JBQVFFLElBQVIsQ0FBYTtBQUNULDRCQUFRL0QsSUFBSUMsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsTUFBeEIsRUFBZ0MsU0FBaEMsQ0FEQztBQUVULDZCQUFTLGlCQUZBO0FBR1QsNkJBQVMsZUFBVVIsS0FBVixFQUFpQjtBQUN0QjtBQUNBO0FBQ0FDLDhCQUFNb0UsTUFBTjtBQUNIO0FBUFEsaUJBQWI7QUFTQTtBQUNKLGlCQUFLLGVBQUw7QUFDSUgsd0JBQVFFLElBQVIsQ0FBYTtBQUNULDRCQUFRL0QsSUFBSUMsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsTUFBeEIsRUFBZ0MsU0FBaEMsQ0FEQztBQUVULDZCQUFTLHVDQUZBO0FBR1QsNkJBQVMsZUFBVVIsS0FBVixFQUFpQjtBQUN0QkEsOEJBQU1HLGNBQU47O0FBRUEsNEJBQUllLE1BQU0sMEJBQTBCakIsTUFBTVosSUFBTixDQUFXLFNBQVgsQ0FBMUIsR0FDSixzQ0FETjtBQUVBLDRCQUFJQSxPQUFPWSxNQUFNaUMsSUFBTixDQUFXLE1BQVgsRUFBbUJxQyxTQUFuQixFQUFYOztBQUVBaEYsMEJBQUU0RCxJQUFGLENBQU8sRUFBQ2pDLFFBQUQsRUFBTTdCLFVBQU4sRUFBWW1GLE1BQU0sTUFBbEIsRUFBMEJwQixVQUFVLE1BQXBDLEVBQVAsRUFBb0Q1QixJQUFwRCxDQUF5RCxVQUFDQyxRQUFELEVBQWM7QUFDbkVsQyw4QkFBRSxxQkFBRixFQUF5QnVELElBQXpCO0FBQ0F2RCw4QkFBRSx3QkFBRixFQUE0QnNELElBQTVCOztBQUVBdEQsOEJBQUUsd0JBQUYsRUFBNEJ1RCxJQUE1QjtBQUNILHlCQUxEO0FBTUg7QUFoQlEsaUJBQWI7QUF6Q1I7O0FBNkRBLGVBQU9vQixPQUFQO0FBQ0gsS0F4RUQ7O0FBMEVBLFFBQUlJLCtCQUErQixTQUEvQkEsNEJBQStCLENBQVV0RSxLQUFWLEVBQWlCO0FBQ2hEQSxjQUFNeUUsZUFBTjs7QUFFQSxZQUFJeEQsZ0JBQWdCMUIsRUFBRSwrQkFBRixFQUFtQzBELEdBQW5DLEVBQXBCO0FBQ0EsWUFBSWhDLGtCQUFrQixFQUF0QixFQUEwQjtBQUN0QixtQkFBTyxLQUFQO0FBQ0g7O0FBRUQxQixVQUFFNEQsSUFBRixDQUFPO0FBQ0gsb0JBQVEsTUFETDtBQUVILG1CQUFPLGlFQUZKO0FBR0gsdUJBQVcsS0FIUjtBQUlILHdCQUFZLE1BSlQ7QUFLSCx1QkFBVyxJQUxSO0FBTUgsb0JBQVE7O0FBRUosaUNBQWlCbEMsYUFGYjtBQUdKLDhCQUFjMUIsRUFBRSwyQ0FBRixFQUErQzBELEdBQS9DLEVBSFY7QUFJSiw0QkFBWTNELE1BQU1ELElBQU4sQ0FBVyxVQUFYLENBSlI7QUFLSiw4QkFBY0UsRUFBRSxhQUFGLEVBQWlCMEQsR0FBakI7QUFMVixhQU5MO0FBYUh6QixrQkFBTSxnQkFBWTtBQUNka0QseUJBQVNDLFFBQVQsQ0FBa0JDLE1BQWxCO0FBQ0g7QUFmRSxTQUFQOztBQWtCQSxlQUFPLEtBQVA7QUFDSCxLQTNCRDs7QUE2QkEsUUFBSTdDLHNCQUFzQixTQUF0QkEsbUJBQXNCLENBQVU5QixLQUFWLEVBQWlCO0FBQ3ZDLFlBQUk0RSxXQUFXLEVBQWY7O0FBRUFwRixvQkFBWW9DLE1BQVosQ0FBbUIsVUFBbkIsRUFBK0JpRCxJQUEvQixDQUFvQyxZQUFZO0FBQzVDN0Usa0JBQU1nQyxNQUFOLENBQWEsMERBQTBEMUMsRUFBRSxJQUFGLEVBQVEwRCxHQUFSLEVBQTFELEdBQ1QsTUFESjs7QUFHQTRCLHFCQUFTVCxJQUFULENBQWM3RSxFQUFFLElBQUYsRUFBUTBELEdBQVIsRUFBZDtBQUNILFNBTEQ7O0FBT0FoRCxjQUFNaUMsSUFBTixDQUFXLGtCQUFYLEVBQStCQyxJQUEvQixDQUFvQzBDLFNBQVNFLElBQVQsQ0FBYyxJQUFkLENBQXBDO0FBQ0gsS0FYRDs7QUFhQTtBQUNBO0FBQ0E7O0FBRUEzRixXQUFPMEUsSUFBUCxHQUFjLFVBQVV0QyxJQUFWLEVBQWdCO0FBQzFCLGdCQUFRM0IsUUFBUXNFLE1BQWhCO0FBQ0ksaUJBQUssUUFBTDtBQUNJN0Usc0JBQU15RSxFQUFOLENBQVMsT0FBVCxFQUFrQmhFLGlCQUFsQjtBQUNBO0FBQ0osaUJBQUssY0FBTDtBQUNJVCxzQkFBTXlFLEVBQU4sQ0FBUyxPQUFULEVBQWtCcEMsc0JBQWxCO0FBQ0E7QUFDSixpQkFBSyxtQkFBTDtBQUNJckMsc0JBQU15RSxFQUFOLENBQVMsT0FBVCxFQUFrQjFCLHVCQUFsQjtBQUNBO0FBQ0osaUJBQUssc0JBQUw7QUFDSS9DLHNCQUFNeUUsRUFBTixDQUFTLE9BQVQsRUFBa0IzQiw2QkFBbEI7QUFDQTtBQUNKLGlCQUFLLGNBQUw7QUFDSTlDLHNCQUFNeUUsRUFBTixDQUFTLE9BQVQsRUFBa0IvQixzQkFBbEI7QUFDQTtBQUNKLGlCQUFLLGVBQUw7QUFDSTFDLHNCQUFNeUUsRUFBTixDQUFTLE9BQVQsRUFBa0J6Qix1QkFBbEI7QUFDQTtBQWxCUjs7QUFxQkEsWUFBSXpDLFFBQVFtRixTQUFSLEtBQXNCLHVCQUExQixFQUFtRDtBQUMvQzFGLGtCQUFNeUUsRUFBTixDQUFTLE9BQVQsRUFBa0IsYUFBbEIsRUFBaUNuRCw2QkFBakM7QUFDSDs7QUFFRFk7QUFDSCxLQTNCRDs7QUE2QkEsV0FBT3BDLE1BQVA7QUFDSCxDQW5kTCIsImZpbGUiOiJvcmRlcnMvb3JkZXJzX21vZGFsX2xheWVyLmpzIiwic291cmNlc0NvbnRlbnQiOlsiLyogLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbiBvcmRlcnNfbW9kYWxfbGF5ZXIuanMgMjAxNi0wMy0xNlxuIEdhbWJpbyBHbWJIXG4gaHR0cDovL3d3dy5nYW1iaW8uZGVcbiBDb3B5cmlnaHQgKGMpIDIwMTYgR2FtYmlvIEdtYkhcbiBSZWxlYXNlZCB1bmRlciB0aGUgR05VIEdlbmVyYWwgUHVibGljIExpY2Vuc2UgKFZlcnNpb24gMilcbiBbaHR0cDovL3d3dy5nbnUub3JnL2xpY2Vuc2VzL2dwbC0yLjAuaHRtbF1cbiAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICovXG5cbi8qKlxuICogIyMgT3JkZXJzIE1vZGFsIExheWVyIE1vZHVsZVxuICpcbiAqIFRoaXMgbW9kdWxlIHdpbGwgb3BlbiBhIG1vZGFsIGxheWVyIGZvciBvcmRlciBhY3Rpb25zIGxpa2UgZGVsZXRpbmcgb3IgY2hhbmdpbmcgdGhlIG9kZXIgc3RhdHVzLlxuICpcbiAqIEBtb2R1bGUgQ29tcGF0aWJpbGl0eS9vcmRlcnNfbW9kYWxfbGF5ZXJcbiAqL1xuZ3guY29tcGF0aWJpbGl0eS5tb2R1bGUoXG4gICAgJ29yZGVyc19tb2RhbF9sYXllcicsXG5cbiAgICBbJ3hocicsICdmYWxsYmFjayddLFxuXG4gICAgLyoqICBAbGVuZHMgbW9kdWxlOkNvbXBhdGliaWxpdHkvb3JkZXJzX21vZGFsX2xheWVyICovXG5cbiAgICBmdW5jdGlvbiAoZGF0YSkge1xuXG4gICAgICAgICd1c2Ugc3RyaWN0JztcblxuICAgICAgICAvLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbiAgICAgICAgLy8gVkFSSUFCTEVTIERFRklOSVRJT05cbiAgICAgICAgLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG5cbiAgICAgICAgdmFyXG4gICAgICAgICAgICAvKipcbiAgICAgICAgICAgICAqIE1vZHVsZSBTZWxlY3RvclxuICAgICAgICAgICAgICpcbiAgICAgICAgICAgICAqIEB2YXIge29iamVjdH1cbiAgICAgICAgICAgICAqL1xuICAgICAgICAgICAgJHRoaXMgPSAkKHRoaXMpLFxuXG4gICAgICAgICAgICAvKipcbiAgICAgICAgICAgICAqIE1vZGFsIFNlbGVjdG9yXG4gICAgICAgICAgICAgKlxuICAgICAgICAgICAgICogQHR5cGUge29iamVjdH1cbiAgICAgICAgICAgICAqL1xuICAgICAgICAgICAgJG1vZGFsID0gJCgnI21vZGFsX2xheWVyX2NvbnRhaW5lcicpLFxuXG4gICAgICAgICAgICAvKipcbiAgICAgICAgICAgICAqIENoZWNrYm94ZXMgU2VsZWN0b3JcbiAgICAgICAgICAgICAqXG4gICAgICAgICAgICAgKiBAdHlwZSB7b2JqZWN0fVxuICAgICAgICAgICAgICovXG4gICAgICAgICAgICAkY2hlY2tib3hlcyA9ICQoJy5neC1vcmRlcnMtdGFibGUgdHI6bm90KC5kYXRhVGFibGVIZWFkaW5nUm93KSBpbnB1dCcpLFxuXG4gICAgICAgICAgICAvKipcbiAgICAgICAgICAgICAqIERlZmF1bHQgT3B0aW9uc1xuICAgICAgICAgICAgICpcbiAgICAgICAgICAgICAqIEB0eXBlIHtvYmplY3R9XG4gICAgICAgICAgICAgKi9cbiAgICAgICAgICAgIGRlZmF1bHRzID0ge1xuICAgICAgICAgICAgICAgIGRldGFpbF9wYWdlOiBmYWxzZSxcbiAgICAgICAgICAgICAgICBjb21tZW50OiAnJ1xuICAgICAgICAgICAgfSxcblxuICAgICAgICAgICAgLyoqXG4gICAgICAgICAgICAgKiBGaW5hbCBPcHRpb25zXG4gICAgICAgICAgICAgKlxuICAgICAgICAgICAgICogQHZhciB7b2JqZWN0fVxuICAgICAgICAgICAgICovXG4gICAgICAgICAgICBvcHRpb25zID0gJC5leHRlbmQodHJ1ZSwge30sIGRlZmF1bHRzLCBkYXRhKSxcblxuICAgICAgICAgICAgLyoqXG4gICAgICAgICAgICAgKiBNb2R1bGUgT2JqZWN0XG4gICAgICAgICAgICAgKlxuICAgICAgICAgICAgICogQHR5cGUge29iamVjdH1cbiAgICAgICAgICAgICAqL1xuICAgICAgICAgICAgbW9kdWxlID0ge307XG5cbiAgICAgICAgLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4gICAgICAgIC8vIFBSSVZBVEUgRlVOQ1RJT05TXG4gICAgICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuXG4gICAgICAgIHZhciBfb3BlbkRlbGV0ZURpYWxvZyA9IGZ1bmN0aW9uIChldmVudCkge1xuXG4gICAgICAgICAgICB2YXIgJGZvcm0gPSAkKCcjZGVsZXRlX2NvbmZpcm1fZm9ybScpO1xuICAgICAgICAgICAgJGZvcm0uYXR0cignYWN0aW9uJywgJGZvcm0uYXR0cignYWN0aW9uJykgKyAnJm9JRD0nICsgJHRoaXMuZGF0YSgnb3JkZXJfaWQnKSk7XG5cbiAgICAgICAgICAgIGV2ZW50LnByZXZlbnREZWZhdWx0KCk7XG5cbiAgICAgICAgICAgIHZhciB0aXRsZSA9IGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdURVhUX0lORk9fSEVBRElOR19ERUxFVEVfT1JERVInLCAnb3JkZXJzJylcbiAgICAgICAgICAgICAgICAucmVwbGFjZSgnJXMnLCAkdGhpcy5kYXRhKCdvcmRlcl9pZCcpKTtcblxuICAgICAgICAgICAgJGZvcm0uZGlhbG9nKHtcbiAgICAgICAgICAgICAgICAndGl0bGUnOiB0aXRsZSxcbiAgICAgICAgICAgICAgICAnbW9kYWwnOiB0cnVlLFxuICAgICAgICAgICAgICAgICdkaWFsb2dDbGFzcyc6ICdneC1jb250YWluZXInLFxuICAgICAgICAgICAgICAgICdidXR0b25zJzogX2dldE1vZGFsQnV0dG9ucygkZm9ybSksXG4gICAgICAgICAgICAgICAgJ3dpZHRoJzogNDIwXG4gICAgICAgICAgICB9KTtcblxuICAgICAgICB9O1xuXG4gICAgICAgIHZhciBfb3BlblRyYWNraW5nQ29kZURlbGV0ZURpYWxvZyA9IGZ1bmN0aW9uIChldmVudCkge1xuICAgICAgICAgICAgdmFyICRmb3JtID0gJCgnI2RlbGV0ZV90cmFja2luZ19jb2RlX2NvbmZpcm1fZm9ybScpO1xuICAgICAgICAgICAgdmFyIGRhdGFfc2V0ID0ganNlLmxpYnMuZmFsbGJhY2suX2RhdGEoJCh0aGlzKSwgJ29yZGVyc19tb2RhbF9sYXllcicpO1xuICAgICAgICAgICAgJGZvcm0uZGlhbG9nKHtcbiAgICAgICAgICAgICAgICAndGl0bGUnOiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnVFhUX1BBUkNFTF9UUkFDS0lOR19ERUxFVEVfQlVUVE9OJywgJ3BhcmNlbF9zZXJ2aWNlcycpXG4gICAgICAgICAgICAgICAgICAgIC5yZXBsYWNlKFxuICAgICAgICAgICAgICAgICAgICAgICAgJyVzJywgZGF0YV9zZXQudHJhY2tpbmdfY29kZSksXG4gICAgICAgICAgICAgICAgJ21vZGFsJzogdHJ1ZSxcbiAgICAgICAgICAgICAgICAnZGlhbG9nQ2xhc3MnOiAnZ3gtY29udGFpbmVyJyxcbiAgICAgICAgICAgICAgICAnYnV0dG9ucyc6IFtcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgJ3RleHQnOiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnY2xvc2UnLCAnYnV0dG9ucycpLFxuICAgICAgICAgICAgICAgICAgICAgICAgJ2NsYXNzJzogJ2J0bicsXG4gICAgICAgICAgICAgICAgICAgICAgICAnY2xpY2snOiBmdW5jdGlvbiAoKSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgJCh0aGlzKS5kaWFsb2coJ2Nsb3NlJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICd0ZXh0JzoganNlLmNvcmUubGFuZy50cmFuc2xhdGUoJ2RlbGV0ZScsICdidXR0b25zJyksXG4gICAgICAgICAgICAgICAgICAgICAgICAnY2xhc3MnOiAnYnRuIGJ0bi1wcmltYXJ5JyxcbiAgICAgICAgICAgICAgICAgICAgICAgICdjbGljayc6IGZ1bmN0aW9uICgpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAkKHRoaXMpLmRpYWxvZygnY2xvc2UnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIHZhciB1cmwgPSAncmVxdWVzdF9wb3J0LnBocD9tb2R1bGU9UGFyY2VsU2VydmljZXMmYWN0aW9uPWRlbGV0ZV90cmFja2luZ19jb2RlJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIGpzZS5saWJzLnhoci5wb3N0KHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgJ3VybCc6IHVybCxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgJ2RhdGEnOiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAndHJhY2tpbmdfY29kZV9pZCc6IGRhdGFfc2V0LnRyYWNraW5nX2NvZGVfaWQsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnb3JkZXJfaWQnOiBkYXRhX3NldC5vcmRlcl9pZCxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdwYWdlX3Rva2VuJzogZGF0YV9zZXQucGFnZV90b2tlblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgfSkuZG9uZShmdW5jdGlvbiAocmVzcG9uc2UpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgJCgnI3RyYWNraW5nX2NvZGVfd3JhcHBlciA+IC5mcmFtZS1jb250ZW50ID4gdGFibGUnKS5odG1sKHJlc3BvbnNlLmh0bWwpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAnd2lkdGgnOiA0MjBcbiAgICAgICAgICAgIH0pO1xuXG4gICAgICAgIH07XG5cbiAgICAgICAgdmFyIF9vcGVuTXVsdGlEZWxldGVEaWFsb2cgPSBmdW5jdGlvbiAoZXZlbnQpIHtcblxuICAgICAgICAgICAgdmFyICRmb3JtID0gJCgnI211bHRpX2RlbGV0ZV9jb25maXJtX2Zvcm0nKSxcbiAgICAgICAgICAgICAgICBvcmRlcklkID0gMDtcblxuICAgICAgICAgICAgZXZlbnQucHJldmVudERlZmF1bHQoKTtcblxuICAgICAgICAgICAgaWYgKCRjaGVja2JveGVzLmZpbHRlcignOmNoZWNrZWQnKS5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgICAgICByZXR1cm4gZmFsc2U7XG4gICAgICAgICAgICB9XG5cbiAgICAgICAgICAgIF9yZWFkU2VsZWN0ZWRPcmRlcnMoJGZvcm0pO1xuXG4gICAgICAgICAgICAkZm9ybS5hdHRyKCdhY3Rpb24nLCAkZm9ybS5hdHRyKCdhY3Rpb24nKSArICcmb0lEPScgKyAkdGhpcy5kYXRhKCdvcmRlcl9pZCcpKTtcblxuICAgICAgICAgICAgJGZvcm0uZGlhbG9nKHtcbiAgICAgICAgICAgICAgICAndGl0bGUnOiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnVEVYVF9JTkZPX0hFQURJTkdfTVVMVElfREVMRVRFX09SREVSJywgJ29yZGVycycpXG4gICAgICAgICAgICAgICAgICAgIC5yZXBsYWNlKCclcycsXG4gICAgICAgICAgICAgICAgICAgICAgICAkdGhpcy5kYXRhKCdvcmRlcl9pZCcpKSxcbiAgICAgICAgICAgICAgICAnbW9kYWwnOiB0cnVlLFxuICAgICAgICAgICAgICAgICdkaWFsb2dDbGFzcyc6ICdneC1jb250YWluZXInLFxuICAgICAgICAgICAgICAgICdidXR0b25zJzogX2dldE1vZGFsQnV0dG9ucygkZm9ybSksXG4gICAgICAgICAgICAgICAgJ3dpZHRoJzogNDIwXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfTtcblxuICAgICAgICB2YXIgX29wZW5NdWx0aUNhbmNlbERpYWxvZyA9IGZ1bmN0aW9uIChldmVudCkge1xuICAgICAgICAgICAgdmFyICRmb3JtID0gJCgnI211bHRpX2NhbmNlbF9jb25maXJtX2Zvcm0nKTtcbiAgICAgICAgICAgIGV2ZW50LnByZXZlbnREZWZhdWx0KCk7XG5cbiAgICAgICAgICAgIGlmIChvcHRpb25zLmRldGFpbF9wYWdlKSB7XG4gICAgICAgICAgICAgICAgLy8gT3JkZXJzIGRldGFpbCBwYWdlXG4gICAgICAgICAgICAgICAgJGZvcm0uYXBwZW5kKCc8aW5wdXQgdHlwZT1cImhpZGRlblwiIG5hbWU9XCJnbV9tdWx0aV9zdGF0dXNbXVwiIHZhbHVlPVwiJyArIG9wdGlvbnMub3JkZXJfaWQgK1xuICAgICAgICAgICAgICAgICAgICAnXCIgLz4nKTtcbiAgICAgICAgICAgICAgICAkZm9ybS5maW5kKCcuc2VsZWN0ZWRfb3JkZXJzJykudGV4dChvcHRpb25zLm9yZGVyX2lkKTtcbiAgICAgICAgICAgICAgICAkZm9ybS5maW5kKCd0ZXh0YXJlYVtuYW1lPVwiZ21fY29tbWVudHNcIl0nKS5odG1sKG9wdGlvbnMuY29tbWVudCk7XG4gICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICAgIC8vIE9yZGVycyBwYWdlXG4gICAgICAgICAgICAgICAgaWYgKCRjaGVja2JveGVzLmZpbHRlcignOmNoZWNrZWQnKS5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBfcmVhZFNlbGVjdGVkT3JkZXJzKCRmb3JtKTtcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgJGZvcm0uYXR0cignYWN0aW9uJywgJGZvcm0uYXR0cignYWN0aW9uJykgKyAnP29JRD0nICsgJHRoaXMuZGF0YSgnb3JkZXJfaWQnKSArICcmb3JpZ2luPW9sZF9vcmRlcnNfb3ZlcnZpZXcnKTtcblxuICAgICAgICAgICAgJGZvcm0uZGlhbG9nKHtcbiAgICAgICAgICAgICAgICAndGl0bGUnOiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnVEVYVF9JTkZPX0hFQURJTkdfTVVMVElfQ0FOQ0VMX09SREVSJywgJ29yZGVycycpXG4gICAgICAgICAgICAgICAgICAgIC5yZXBsYWNlKCclcycsXG4gICAgICAgICAgICAgICAgICAgICAgICAkdGhpcy5kYXRhKCdvcmRlcl9pZCcpKSxcbiAgICAgICAgICAgICAgICAnbW9kYWwnOiB0cnVlLFxuICAgICAgICAgICAgICAgICdkaWFsb2dDbGFzcyc6ICdneC1jb250YWluZXInLFxuICAgICAgICAgICAgICAgICdidXR0b25zJzogX2dldE1vZGFsQnV0dG9ucygkZm9ybSksXG4gICAgICAgICAgICAgICAgJ3dpZHRoJzogNDIwXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfTtcblxuICAgICAgICB2YXIgX29wZW5VcGRhdGVPcmRlcnNTdGF0dXNEaWFsb2cgPSBmdW5jdGlvbiAoZXZlbnQpIHtcbiAgICAgICAgICAgIHZhciAkZm9ybSA9ICQoJyN1cGRhdGVfb3JkZXJzX3N0YXR1c19mb3JtJyk7XG5cbiAgICAgICAgICAgIGV2ZW50LnByZXZlbnREZWZhdWx0KCk7XG5cbiAgICAgICAgICAgIGlmIChvcHRpb25zLmRldGFpbF9wYWdlKSB7XG4gICAgICAgICAgICAgICAgLy8gT3JkZXJzIGRldGFpbCBwYWdlXG4gICAgICAgICAgICAgICAgJGZvcm0uYXBwZW5kKCc8aW5wdXQgdHlwZT1cImhpZGRlblwiIG5hbWU9XCJnbV9tdWx0aV9zdGF0dXNbXVwiIHZhbHVlPVwiJyArIG9wdGlvbnMub3JkZXJfaWQgK1xuICAgICAgICAgICAgICAgICAgICAnXCIgLz4nKTtcbiAgICAgICAgICAgICAgICAkZm9ybS5maW5kKCcuc2VsZWN0ZWRfb3JkZXJzJykudGV4dChvcHRpb25zLm9yZGVyX2lkKTtcbiAgICAgICAgICAgICAgICAkZm9ybS5maW5kKCd0ZXh0YXJlYVtuYW1lPVwiZ21fY29tbWVudHNcIl0nKS5odG1sKG9wdGlvbnMuY29tbWVudCk7XG4gICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICAgIC8vIE9yZGVycyBwYWdlXG4gICAgICAgICAgICAgICAgaWYgKCRjaGVja2JveGVzLmZpbHRlcignOmNoZWNrZWQnKS5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBfcmVhZFNlbGVjdGVkT3JkZXJzKCRmb3JtKTtcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgJGZvcm0uZGlhbG9nKHtcbiAgICAgICAgICAgICAgICAndGl0bGUnOiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnSEVBRElOR19HTV9TVEFUVVMnLCAnb3JkZXJzJyksXG4gICAgICAgICAgICAgICAgJ21vZGFsJzogdHJ1ZSxcbiAgICAgICAgICAgICAgICAnZGlhbG9nQ2xhc3MnOiAnZ3gtY29udGFpbmVyJyxcbiAgICAgICAgICAgICAgICAnYnV0dG9ucyc6IF9nZXRNb2RhbEJ1dHRvbnMoJGZvcm0pLFxuICAgICAgICAgICAgICAgICd3aWR0aCc6IDU4MFxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH07XG5cbiAgICAgICAgdmFyIF9vcGVuVHJhY2tpbmdDb2RlRGlhbG9nID0gZnVuY3Rpb24gKGV2ZW50KSB7XG5cbiAgICAgICAgICAgIHZhciAkZm9ybSA9ICQoJyNhZGRfdHJhY2tpbmdfY29kZV9mb3JtJyk7XG5cbiAgICAgICAgICAgIGV2ZW50LnByZXZlbnREZWZhdWx0KCk7XG4gICAgICAgICAgICAkZm9ybS5kaWFsb2coe1xuICAgICAgICAgICAgICAgICd0aXRsZSc6IGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdUWFRfUEFSQ0VMX1RSQUNLSU5HX0hFQURJTkcnLCAncGFyY2VsX3NlcnZpY2VzJylcbiAgICAgICAgICAgICAgICAgICAgLnJlcGxhY2UoJyVzJywgJHRoaXMuZGF0YSgnb3JkZXJfaWQnKSksXG4gICAgICAgICAgICAgICAgJ21vZGFsJzogdHJ1ZSxcbiAgICAgICAgICAgICAgICAnZGlhbG9nQ2xhc3MnOiAnZ3gtY29udGFpbmVyJyxcbiAgICAgICAgICAgICAgICAnYnV0dG9ucyc6IF9nZXRNb2RhbEJ1dHRvbnMoJGZvcm0pLFxuICAgICAgICAgICAgICAgICd3aWR0aCc6IDQyMFxuICAgICAgICAgICAgfSk7XG5cbiAgICAgICAgfTtcblxuICAgICAgICB2YXIgX29wZW5FbWFpbEludm9pY2VEaWFsb2cgPSBmdW5jdGlvbiAoZXZlbnQpIHtcblxuICAgICAgICAgICAgY29uc3QgJG1vZGFsID0gJCgnLmVtYWlsLWludm9pY2UtbW9kYWwtYm9keScpO1xuICAgICAgICAgICAgY29uc3QgdXJsID0ganNlLmNvcmUuY29uZmlnLmdldCgnYXBwVXJsJykgKyAnL2FkbWluL2FkbWluLnBocCc7XG4gICAgICAgICAgICBjb25zdCBkYXRhID0ge1xuICAgICAgICAgICAgICAgIGlkOiBvcHRpb25zLm9yZGVyX2lkLFxuICAgICAgICAgICAgICAgIGRvOiAnT3JkZXJzTW9kYWxzQWpheC9HZXRFbWFpbEludm9pY2VTdWJqZWN0JyxcbiAgICAgICAgICAgICAgICBwYWdlVG9rZW46IGpzZS5jb3JlLmNvbmZpZy5nZXQoJ3BhZ2VUb2tlbicpXG4gICAgICAgICAgICB9O1xuICAgICAgICAgICAgbGV0IGludm9pY2VOdW1iZXJzSHRtbCA9ICcnO1xuXG4gICAgICAgICAgICAkKCcuZW1haWwtaW52b2ljZS1mb3JtJykuc2hvdygpO1xuICAgICAgICAgICAgJCgnLmVtYWlsLWludm9pY2Utc3VjY2VzcycpLmhpZGUoKTtcblxuICAgICAgICAgICAgaWYgKCRtb2RhbC5maW5kKCcubWVzc2FnZV9zdGFja19jb250YWluZXInKS5sZW5ndGggPCAxKSB7XG4gICAgICAgICAgICAgICAgJG1vZGFsLmZpbmQoJy5hbGVydCcpLndyYXAoJzxkaXYgY2xhc3M9XCJtZXNzYWdlX3N0YWNrX2NvbnRhaW5lclwiPjwvZGl2PicpO1xuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICAkbW9kYWwuZmluZCgnLmN1c3RvbWVyLWluZm8nKS50ZXh0KGBcIiR7b3B0aW9ucy5uYW1lfVwiYCk7XG4gICAgICAgICAgICAkbW9kYWwuZmluZCgnLmVtYWlsLWFkZHJlc3MnKS52YWwob3B0aW9ucy5lbWFpbCk7XG5cbiAgICAgICAgICAgICRtb2RhbFxuICAgICAgICAgICAgICAgIC5kYXRhKCdvcmRlcklkJywgb3B0aW9ucy5vcmRlcl9pZCk7XG5cbiAgICAgICAgICAgICQuYWpheCh7dXJsLCBkYXRhLCBkYXRhVHlwZTogJ2pzb24nfSkuZG9uZSgocmVzcG9uc2UpID0+IHtcbiAgICAgICAgICAgICAgICAkbW9kYWwuYXR0cignZGF0YS1neC13aWRnZXQnLCAnc2luZ2xlX2NoZWNrYm94Jyk7XG5cbiAgICAgICAgICAgICAgICAkbW9kYWwuZmluZCgnLnN1YmplY3QnKS52YWwocmVzcG9uc2Uuc3ViamVjdCk7XG4gICAgICAgICAgICAgICAgaWYgKHJlc3BvbnNlLmludm9pY2VJZEV4aXN0cykge1xuICAgICAgICAgICAgICAgICAgICAkbW9kYWwuZmluZCgnLmludm9pY2UtbnVtLWluZm8nKS5hZGRDbGFzcygnaGlkZGVuJyk7XG4gICAgICAgICAgICAgICAgICAgICRtb2RhbC5maW5kKCcubm8taW52b2ljZScpLnJlbW92ZUNsYXNzKCdoaWRkZW4nKTtcbiAgICAgICAgICAgICAgICAgICAgJG1vZGFsLmZpbmQoJy5lbWFpbC1pbnZvaWNlLWZvcm0gLm1lc3NhZ2Vfc3RhY2tfY29udGFpbmVyJykucmVtb3ZlQ2xhc3MoJ2hpZGRlbicpO1xuICAgICAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgICAgICAgICRtb2RhbC5maW5kKCcuaW52b2ljZS1udW0taW5mbycpLnJlbW92ZUNsYXNzKCdoaWRkZW4nKTtcbiAgICAgICAgICAgICAgICAgICAgJG1vZGFsLmZpbmQoJy5uby1pbnZvaWNlJykuYWRkQ2xhc3MoJ2hpZGRlbicpO1xuICAgICAgICAgICAgICAgICAgICAkbW9kYWwuZmluZCgnLmVtYWlsLWludm9pY2UtZm9ybSAubWVzc2FnZV9zdGFja19jb250YWluZXInKS5hZGRDbGFzcygnaGlkZGVuJyk7XG4gICAgICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICAgICAgaWYgKE9iamVjdC5rZXlzKHJlc3BvbnNlLmludm9pY2VOdW1iZXJzKS5sZW5ndGggPD0gMSkge1xuICAgICAgICAgICAgICAgICAgICAkbW9kYWwuZmluZCgnLmludm9pY2UtbnVtYmVycycpLmFkZENsYXNzKCdoaWRkZW4nKTtcbiAgICAgICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICAgICAgICAkbW9kYWwuZmluZCgnLmludm9pY2UtbnVtYmVycycpLnJlbW92ZUNsYXNzKCdoaWRkZW4nKTtcbiAgICAgICAgICAgICAgICB9XG5cbiAgICAgICAgICAgICAgICBmb3IgKGxldCBpbnZvaWNlSWQgaW4gcmVzcG9uc2UuaW52b2ljZU51bWJlcnMpIHtcbiAgICAgICAgICAgICAgICAgICAgaW52b2ljZU51bWJlcnNIdG1sICs9XG4gICAgICAgICAgICAgICAgICAgICAgICAnPHA+PGlucHV0IHR5cGU9XCJjaGVja2JveFwiIG5hbWU9XCJpbnZvaWNlX2lkc1tdXCIgdmFsdWU9XCInICsgaW52b2ljZUlkXG4gICAgICAgICAgICAgICAgICAgICAgICArICdcIiBjaGVja2VkPVwiY2hlY2tlZFwiIGNsYXNzPVwiaW52b2ljZS1udW1iZXJzLWNoZWNrYm94XCIgLz4gJ1xuICAgICAgICAgICAgICAgICAgICAgICAgKyByZXNwb25zZS5pbnZvaWNlTnVtYmVyc1tpbnZvaWNlSWRdICsgJzwvcD4nO1xuICAgICAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgICAgICRtb2RhbC5maW5kKCcuaW52b2ljZS1udW1iZXJzLWNoZWNrYm94ZXMnKS5odG1sKGludm9pY2VOdW1iZXJzSHRtbCk7XG5cbiAgICAgICAgICAgICAgICBneC53aWRnZXRzLmluaXQoJG1vZGFsKTtcblxuICAgICAgICAgICAgICAgICRtb2RhbC5kaWFsb2coe1xuICAgICAgICAgICAgICAgICAgICAndGl0bGUnOiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnVElUTEVfSU5WT0lDRScsICdnbV9vcmRlcl9tZW51JyksXG4gICAgICAgICAgICAgICAgICAgICdtb2RhbCc6IHRydWUsXG4gICAgICAgICAgICAgICAgICAgICdkaWFsb2dDbGFzcyc6ICdneC1jb250YWluZXInLFxuICAgICAgICAgICAgICAgICAgICAnYnV0dG9ucyc6IF9nZXRNb2RhbEJ1dHRvbnMoJG1vZGFsKSxcbiAgICAgICAgICAgICAgICAgICAgJ3dpZHRoJzogNjAwXG4gICAgICAgICAgICAgICAgfSk7XG5cbiAgICAgICAgICAgICAgICAkbW9kYWwuZmluZCgnLmludm9pY2UtbnVtYmVycy1jaGVja2JveCcpLm9uKCdjaGFuZ2UnLCBmdW5jdGlvbiAoKSB7XG4gICAgICAgICAgICAgICAgICAgIF9vbkNoYW5nZUVtYWlsSW52b2ljZUNoZWNrYm94KCRtb2RhbClcbiAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIH0pO1xuXG4gICAgICAgICAgICBldmVudC5wcmV2ZW50RGVmYXVsdCgpO1xuICAgICAgICB9O1xuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiBPbiBFbWFpbCBJbnZvaWNlIENoZWNrYm94IENoYW5nZVxuICAgICAgICAgKlxuICAgICAgICAgKiBEaXNhYmxlIHNlbmQgYnV0dG9uIGlmIGFsbCBpbnZvaWNlIG51bWJlciBjaGVja2JveGVzIGFyZSB1bmNoZWNrZWQuIE90aGVyd2lzZSBlbmFibGUgdGhlIHNlbmQgYnV0dG9uIGFnYWluLlxuICAgICAgICAgKi9cbiAgICAgICAgdmFyIF9vbkNoYW5nZUVtYWlsSW52b2ljZUNoZWNrYm94ID0gZnVuY3Rpb24gKCRtb2RhbCkge1xuICAgICAgICAgICAgaWYgKCRtb2RhbC5maW5kKCcuaW52b2ljZS1udW1iZXJzLWNoZWNrYm94JykubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgICAgIGlmICgkbW9kYWwuZmluZCgnLmludm9pY2UtbnVtYmVycy1jaGVja2JveDpjaGVja2VkJykubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgICAgICAgICAkKCcuYnRuLXNlbmQtaW52b2ljZS1tYWlsJykucHJvcCgnZGlzYWJsZWQnLCBmYWxzZSk7XG4gICAgICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgICAgICAgJCgnLmJ0bi1zZW5kLWludm9pY2UtbWFpbCcpLnByb3AoJ2Rpc2FibGVkJywgdHJ1ZSk7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgICAkKCcuYnRuLXNlbmQtaW52b2ljZS1tYWlsJykucHJvcCgnZGlzYWJsZWQnLCBmYWxzZSk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH07XG5cbiAgICAgICAgdmFyIF9nZXRNb2RhbEJ1dHRvbnMgPSBmdW5jdGlvbiAoJGZvcm0pIHtcbiAgICAgICAgICAgIHZhciBidXR0b25zID0gW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ3RleHQnOiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnY2xvc2UnLCAnYnV0dG9ucycpLFxuICAgICAgICAgICAgICAgICAgICAnY2xhc3MnOiAnYnRuJyxcbiAgICAgICAgICAgICAgICAgICAgJ2NsaWNrJzogZnVuY3Rpb24gKCkge1xuICAgICAgICAgICAgICAgICAgICAgICAgJCh0aGlzKS5kaWFsb2coJ2Nsb3NlJyk7XG4gICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICBdO1xuICAgICAgICAgICAgc3dpdGNoIChvcHRpb25zLmFjdGlvbikge1xuICAgICAgICAgICAgICAgIGNhc2UgJ2RlbGV0ZSc6XG4gICAgICAgICAgICAgICAgY2FzZSAnbXVsdGlfZGVsZXRlJzpcbiAgICAgICAgICAgICAgICAgICAgYnV0dG9ucy5wdXNoKHtcbiAgICAgICAgICAgICAgICAgICAgICAgICd0ZXh0JzoganNlLmNvcmUubGFuZy50cmFuc2xhdGUoJ2RlbGV0ZScsICdidXR0b25zJyksXG4gICAgICAgICAgICAgICAgICAgICAgICAnY2xhc3MnOiAnYnRuIGJ0bi1wcmltYXJ5JyxcbiAgICAgICAgICAgICAgICAgICAgICAgICdjbGljayc6IGZ1bmN0aW9uICgpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAkZm9ybS5zdWJtaXQoKTtcbiAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICAgICAgICAgIGNhc2UgJ2FkZF90cmFja2luZ19jb2RlJzpcbiAgICAgICAgICAgICAgICAgICAgYnV0dG9ucy5wdXNoKHtcbiAgICAgICAgICAgICAgICAgICAgICAgICd0ZXh0JzoganNlLmNvcmUubGFuZy50cmFuc2xhdGUoJ2FkZCcsICdidXR0b25zJyksXG4gICAgICAgICAgICAgICAgICAgICAgICAnY2xhc3MnOiAnYnRuIGJ0bi1wcmltYXJ5JyxcbiAgICAgICAgICAgICAgICAgICAgICAgICdjbGljayc6IGZ1bmN0aW9uIChldmVudCkge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIF9hZGRUcmFja2luZ0NvZGVGcm9tT3ZlcnZpZXcoZXZlbnQpO1xuICAgICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICAgICAgY2FzZSAndXBkYXRlX29yZGVyc19zdGF0dXMnOlxuICAgICAgICAgICAgICAgICAgICBidXR0b25zLnB1c2goe1xuICAgICAgICAgICAgICAgICAgICAgICAgJ3RleHQnOiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnZXhlY3V0ZScsICdidXR0b25zJyksXG4gICAgICAgICAgICAgICAgICAgICAgICAnY2xhc3MnOiAnYnRuIGJ0bi1wcmltYXJ5JyxcbiAgICAgICAgICAgICAgICAgICAgICAgICdjbGljayc6IGZ1bmN0aW9uIChldmVudCkge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICRmb3JtLnN1Ym1pdCgpO1xuICAgICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICAgICAgY2FzZSAnbXVsdGlfY2FuY2VsJzpcbiAgICAgICAgICAgICAgICAgICAgYnV0dG9ucy5wdXNoKHtcbiAgICAgICAgICAgICAgICAgICAgICAgICd0ZXh0JzoganNlLmNvcmUubGFuZy50cmFuc2xhdGUoJ3NlbmQnLCAnYnV0dG9ucycpLFxuICAgICAgICAgICAgICAgICAgICAgICAgJ2NsYXNzJzogJ2J0biBidG4tcHJpbWFyeScsXG4gICAgICAgICAgICAgICAgICAgICAgICAnY2xpY2snOiBmdW5jdGlvbiAoZXZlbnQpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAvL2V2ZW50LnByZXZlbnREZWZhdWx0KCk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgLy9nbV9jYW5jZWwoJ2dtX3NlbmRfb3JkZXIucGhwJywgJyZ0eXBlPWNhbmNlbCcsICdDQU5DRUwnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAkZm9ybS5zdWJtaXQoKTtcbiAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICAgICAgICAgIGNhc2UgJ2VtYWlsX2ludm9pY2UnOlxuICAgICAgICAgICAgICAgICAgICBidXR0b25zLnB1c2goe1xuICAgICAgICAgICAgICAgICAgICAgICAgJ3RleHQnOiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnc2VuZCcsICdidXR0b25zJyksXG4gICAgICAgICAgICAgICAgICAgICAgICAnY2xhc3MnOiAnYnRuIGJ0bi1wcmltYXJ5IGJ0bi1zZW5kLWludm9pY2UtbWFpbCcsXG4gICAgICAgICAgICAgICAgICAgICAgICAnY2xpY2snOiBmdW5jdGlvbiAoZXZlbnQpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBldmVudC5wcmV2ZW50RGVmYXVsdCgpO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgdmFyIHVybCA9ICdnbV9wZGZfb3JkZXIucGhwP29JRD0nICsgJGZvcm0uZGF0YSgnb3JkZXJJZCcpXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICsgJyZ0eXBlPWludm9pY2UmbWFpbD0xJmdtX3F1aWNrX21haWw9MSc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgdmFyIGRhdGEgPSAkZm9ybS5maW5kKCdmb3JtJykuc2VyaWFsaXplKCk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAkLmFqYXgoe3VybCwgZGF0YSwgdHlwZTogJ1BPU1QnLCBkYXRhVHlwZTogJ2h0bWwnfSkuZG9uZSgocmVzcG9uc2UpID0+IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgJCgnLmVtYWlsLWludm9pY2UtZm9ybScpLmhpZGUoKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgJCgnLmVtYWlsLWludm9pY2Utc3VjY2VzcycpLnNob3coKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAkKCcuYnRuLXNlbmQtaW52b2ljZS1tYWlsJykuaGlkZSgpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgcmV0dXJuIGJ1dHRvbnM7XG4gICAgICAgIH07XG5cbiAgICAgICAgdmFyIF9hZGRUcmFja2luZ0NvZGVGcm9tT3ZlcnZpZXcgPSBmdW5jdGlvbiAoZXZlbnQpIHtcbiAgICAgICAgICAgIGV2ZW50LnN0b3BQcm9wYWdhdGlvbigpO1xuXG4gICAgICAgICAgICB2YXIgdHJhY2tpbmdfY29kZSA9ICQoJyNwYXJjZWxfc2VydmljZV90cmFja2luZ19jb2RlJykudmFsKCk7XG4gICAgICAgICAgICBpZiAodHJhY2tpbmdfY29kZSA9PT0gJycpIHtcbiAgICAgICAgICAgICAgICByZXR1cm4gZmFsc2U7XG4gICAgICAgICAgICB9XG5cbiAgICAgICAgICAgICQuYWpheCh7XG4gICAgICAgICAgICAgICAgJ3R5cGUnOiAnUE9TVCcsXG4gICAgICAgICAgICAgICAgJ3VybCc6ICdyZXF1ZXN0X3BvcnQucGhwP21vZHVsZT1QYXJjZWxTZXJ2aWNlcyZhY3Rpb249YWRkX3RyYWNraW5nX2NvZGUnLFxuICAgICAgICAgICAgICAgICd0aW1lb3V0JzogMzAwMDAsXG4gICAgICAgICAgICAgICAgJ2RhdGFUeXBlJzogJ2pzb24nLFxuICAgICAgICAgICAgICAgICdjb250ZXh0JzogdGhpcyxcbiAgICAgICAgICAgICAgICAnZGF0YSc6IHtcblxuICAgICAgICAgICAgICAgICAgICAndHJhY2tpbmdfY29kZSc6IHRyYWNraW5nX2NvZGUsXG4gICAgICAgICAgICAgICAgICAgICdzZXJ2aWNlX2lkJzogJCgnI3BhcmNlbF9zZXJ2aWNlc19kcm9wZG93biBvcHRpb246c2VsZWN0ZWQnKS52YWwoKSxcbiAgICAgICAgICAgICAgICAgICAgJ29yZGVyX2lkJzogJHRoaXMuZGF0YSgnb3JkZXJfaWQnKSxcbiAgICAgICAgICAgICAgICAgICAgJ3BhZ2VfdG9rZW4nOiAkKCcucGFnZV90b2tlbicpLnZhbCgpXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBkb25lOiBmdW5jdGlvbiAoKSB7XG4gICAgICAgICAgICAgICAgICAgIGRvY3VtZW50LmxvY2F0aW9uLnJlbG9hZCgpO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH0pO1xuXG4gICAgICAgICAgICByZXR1cm4gZmFsc2U7XG4gICAgICAgIH07XG5cbiAgICAgICAgdmFyIF9yZWFkU2VsZWN0ZWRPcmRlcnMgPSBmdW5jdGlvbiAoJGZvcm0pIHtcbiAgICAgICAgICAgIHZhciBvcmRlcklkcyA9IFtdO1xuXG4gICAgICAgICAgICAkY2hlY2tib3hlcy5maWx0ZXIoJzpjaGVja2VkJykuZWFjaChmdW5jdGlvbiAoKSB7XG4gICAgICAgICAgICAgICAgJGZvcm0uYXBwZW5kKCc8aW5wdXQgdHlwZT1cImhpZGRlblwiIG5hbWU9XCJnbV9tdWx0aV9zdGF0dXNbXVwiIHZhbHVlPVwiJyArICQodGhpcykudmFsKCkgK1xuICAgICAgICAgICAgICAgICAgICAnXCIgLz4nKTtcblxuICAgICAgICAgICAgICAgIG9yZGVySWRzLnB1c2goJCh0aGlzKS52YWwoKSk7XG4gICAgICAgICAgICB9KTtcblxuICAgICAgICAgICAgJGZvcm0uZmluZCgnLnNlbGVjdGVkX29yZGVycycpLnRleHQob3JkZXJJZHMuam9pbignLCAnKSk7XG4gICAgICAgIH07XG5cbiAgICAgICAgLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4gICAgICAgIC8vIElOSVRJQUxJWkFUSU9OXG4gICAgICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuXG4gICAgICAgIG1vZHVsZS5pbml0ID0gZnVuY3Rpb24gKGRvbmUpIHtcbiAgICAgICAgICAgIHN3aXRjaCAob3B0aW9ucy5hY3Rpb24pIHtcbiAgICAgICAgICAgICAgICBjYXNlICdkZWxldGUnOlxuICAgICAgICAgICAgICAgICAgICAkdGhpcy5vbignY2xpY2snLCBfb3BlbkRlbGV0ZURpYWxvZyk7XG4gICAgICAgICAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICAgICAgICAgIGNhc2UgJ211bHRpX2RlbGV0ZSc6XG4gICAgICAgICAgICAgICAgICAgICR0aGlzLm9uKCdjbGljaycsIF9vcGVuTXVsdGlEZWxldGVEaWFsb2cpO1xuICAgICAgICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgICAgICBjYXNlICdhZGRfdHJhY2tpbmdfY29kZSc6XG4gICAgICAgICAgICAgICAgICAgICR0aGlzLm9uKCdjbGljaycsIF9vcGVuVHJhY2tpbmdDb2RlRGlhbG9nKTtcbiAgICAgICAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICAgICAgY2FzZSAndXBkYXRlX29yZGVyc19zdGF0dXMnOlxuICAgICAgICAgICAgICAgICAgICAkdGhpcy5vbignY2xpY2snLCBfb3BlblVwZGF0ZU9yZGVyc1N0YXR1c0RpYWxvZyk7XG4gICAgICAgICAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICAgICAgICAgIGNhc2UgJ211bHRpX2NhbmNlbCc6XG4gICAgICAgICAgICAgICAgICAgICR0aGlzLm9uKCdjbGljaycsIF9vcGVuTXVsdGlDYW5jZWxEaWFsb2cpO1xuICAgICAgICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgICAgICBjYXNlICdlbWFpbF9pbnZvaWNlJzpcbiAgICAgICAgICAgICAgICAgICAgJHRoaXMub24oJ2NsaWNrJywgX29wZW5FbWFpbEludm9pY2VEaWFsb2cpO1xuICAgICAgICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgaWYgKG9wdGlvbnMuY29udGFpbmVyID09PSAndHJhY2tpbmdfY29kZV93cmFwcGVyJykge1xuICAgICAgICAgICAgICAgICR0aGlzLm9uKCdjbGljaycsICcuYnRuLWRlbGV0ZScsIF9vcGVuVHJhY2tpbmdDb2RlRGVsZXRlRGlhbG9nKTtcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgZG9uZSgpO1xuICAgICAgICB9O1xuXG4gICAgICAgIHJldHVybiBtb2R1bGU7XG4gICAgfSk7XG4iXX0=
